<?php
/**
 * Plugin Name: Foldious Image Manager
 * Plugin URI: https://foldious.com
 * Description: Upload and manage images using Foldious storage. Save website images as bucket and manage them through WordPress.
 * Version: 1.0.0
 * Author: Foldious
 * Author URI: https://foldious.com
 * License: GPL v2 or later
 * Text Domain: photoweb
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('PHOTOWEB_PLUGIN_VERSION', '1.0.0');
define('PHOTOWEB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PHOTOWEB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PHOTOWEB_API_BASE_URL', 'https://foldious.com/api/plugin-api'); // Change to your API URL

// Include required files
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-db.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-api.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-admin.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-uploader.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-gallery.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-shortcode.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-gutenberg.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-media-library.php';
require_once PHOTOWEB_PLUGIN_DIR . 'includes/class-photoweb-auto-upload.php';

// Initialize the plugin
class PhotoWeb_Plugin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // AJAX handlers
        add_action('wp_ajax_photoweb_validate_user', array($this, 'ajax_validate_user'));
        add_action('wp_ajax_photoweb_upload_image', array($this, 'ajax_upload_image'));
        add_action('wp_ajax_photoweb_get_images', array($this, 'ajax_get_images'));
        add_action('wp_ajax_photoweb_delete_image', array($this, 'ajax_delete_image'));
        add_action('wp_ajax_photoweb_get_folders', array($this, 'ajax_get_folders'));
        add_action('wp_ajax_photoweb_get_user_details', array($this, 'ajax_get_user_details'));
        add_action('wp_ajax_photoweb_get_user_stats', array($this, 'ajax_get_user_stats'));
        add_action('wp_ajax_photoweb_get_image_details', array($this, 'ajax_get_image_details'));
        add_action('wp_ajax_photoweb_disconnect', array($this, 'ajax_disconnect'));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Foldious',
            'Foldious',
            'manage_options',
            'photoweb',
            array($this, 'render_admin_page'),
            'dashicons-images-alt2',
            30
        );
        
        add_submenu_page(
            'photoweb',
            'Upload Images',
            'Upload Images',
            'manage_options',
            'photoweb-upload',
            array($this, 'render_upload_page')
        );
        
        add_submenu_page(
            'photoweb',
            'Gallery',
            'Gallery',
            'manage_options',
            'photoweb-gallery',
            array($this, 'render_gallery_page')
        );
        
        add_submenu_page(
            'photoweb',
            'Settings',
            'Settings',
            'manage_options',
            'photoweb-settings',
            array($this, 'render_settings_page')
        );
    }
    
    public function register_settings() {
        register_setting('photoweb_settings', 'photoweb_user_email');
        register_setting('photoweb_settings', 'photoweb_user_id');
        register_setting('photoweb_settings', 'photoweb_api_url');
        register_setting('photoweb_settings', 'photoweb_default_folder');
        register_setting('photoweb_settings', 'photoweb_auto_upload', array(
            'type' => 'boolean',
            'default' => false
        ));
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'photoweb') === false) {
            return;
        }
        
        wp_enqueue_style('photoweb-admin-css', PHOTOWEB_PLUGIN_URL . 'assets/css/admin.css', array(), PHOTOWEB_PLUGIN_VERSION);
        wp_enqueue_script('photoweb-admin-js', PHOTOWEB_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), PHOTOWEB_PLUGIN_VERSION, true);
        
        // Get default folder_id
        $default_folder_id = null;
        $api = new PhotoWeb_API();
        $user_details = $api->get_user_details();
        if ($user_details['success'] && isset($user_details['default_folder']['folder_id'])) {
            $default_folder_id = $user_details['default_folder']['folder_id'];
        }
        
        wp_localize_script('photoweb-admin-js', 'photowebData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('photoweb_nonce'),
            'apiUrl' => get_option('photoweb_api_url', PHOTOWEB_API_BASE_URL),
            'userId' => get_option('photoweb_user_id', ''),
            'userEmail' => get_option('photoweb_user_email', ''),
            'defaultFolderId' => $default_folder_id
        ));
    }
    
    public function enqueue_frontend_scripts() {
        wp_enqueue_style('photoweb-frontend-css', PHOTOWEB_PLUGIN_URL . 'assets/css/frontend.css', array(), PHOTOWEB_PLUGIN_VERSION);
        wp_enqueue_script('photoweb-frontend-js', PHOTOWEB_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), PHOTOWEB_PLUGIN_VERSION, true);
    }
    
    public function render_admin_page() {
        $user_email = get_option('photoweb_user_email', '');
        $user_id = get_option('photoweb_user_id', '');
        
        if (empty($user_email) || empty($user_id)) {
            include PHOTOWEB_PLUGIN_DIR . 'templates/setup.php';
        } else {
            include PHOTOWEB_PLUGIN_DIR . 'templates/dashboard.php';
        }
    }
    
    public function render_upload_page() {
        include PHOTOWEB_PLUGIN_DIR . 'templates/upload.php';
    }
    
    public function render_gallery_page() {
        include PHOTOWEB_PLUGIN_DIR . 'templates/gallery.php';
    }
    
    public function render_settings_page() {
        include PHOTOWEB_PLUGIN_DIR . 'templates/settings.php';
    }
    
    // AJAX Handlers
    public function ajax_validate_user() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $email = sanitize_email($_POST['email']);
        
        if (!is_email($email)) {
            wp_send_json_error(array('message' => 'Invalid email address'));
        }
        
        $api = new PhotoWeb_API();
        $result = $api->validate_user($email);
        
        if ($result['success']) {
            update_option('photoweb_user_email', $email);
            update_option('photoweb_user_id', $result['user_id']);
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_upload_image() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        // Get user_id from WordPress options
        $user_id = get_option('photoweb_user_id', '');
        
        if (empty($user_id)) {
            wp_send_json_error(array('message' => 'User not authenticated. Please connect your Foldious account first.'));
        }
        
        // Handle multiple files
        $files = array();
        
        if (!empty($_FILES['images']['name'][0])) {
            // Multiple files uploaded
            $file_count = count($_FILES['images']['name']);
            for ($i = 0; $i < $file_count; $i++) {
                if ($_FILES['images']['error'][$i] === UPLOAD_ERR_OK) {
                    $files[] = array(
                        'name' => $_FILES['images']['name'][$i],
                        'type' => $_FILES['images']['type'][$i],
                        'tmp_name' => $_FILES['images']['tmp_name'][$i],
                        'error' => $_FILES['images']['error'][$i],
                        'size' => $_FILES['images']['size'][$i]
                    );
                }
            }
        } elseif (!empty($_FILES['images']['name'])) {
            // Single file with 'images' name
            if ($_FILES['images']['error'] === UPLOAD_ERR_OK) {
                $files[] = $_FILES['images'];
            }
        } elseif (!empty($_FILES['image'])) {
            // Single file with 'image' name (backward compatibility)
            if ($_FILES['image']['error'] === UPLOAD_ERR_OK) {
                $files[] = $_FILES['image'];
            }
        }
        
        if (empty($files)) {
            wp_send_json_error(array('message' => 'No image files provided or upload error'));
        }
        
        $api = new PhotoWeb_API();
        $result = $api->upload_images($files, isset($_POST['folder_id']) ? intval($_POST['folder_id']) : null);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_get_images() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $folder_id = isset($_POST['folder_id']) ? intval($_POST['folder_id']) : null;
        
        $api = new PhotoWeb_API();
        $result = $api->get_images($page, $folder_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_delete_image() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $image_id = isset($_POST['image_id']) ? intval($_POST['image_id']) : 0;
        
        if (!$image_id) {
            wp_send_json_error(array('message' => 'Invalid image ID'));
        }
        
        $api = new PhotoWeb_API();
        $result = $api->delete_image($image_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_get_folders() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $api = new PhotoWeb_API();
        $result = $api->get_folders();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_get_user_details() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $api = new PhotoWeb_API();
        $result = $api->get_user_details();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_get_user_stats() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $api = new PhotoWeb_API();
        $result = $api->get_user_stats();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_get_image_details() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        $image_id = isset($_POST['image_id']) ? intval($_POST['image_id']) : 0;
        
        if (!$image_id) {
            wp_send_json_error(array('message' => 'Invalid image ID'));
        }
        
        $api = new PhotoWeb_API();
        $result = $api->get_image_details($image_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_disconnect() {
        check_ajax_referer('photoweb_nonce', 'nonce');
        
        delete_option('photoweb_user_email');
        delete_option('photoweb_user_id');
        
        wp_send_json_success(array('message' => 'Account disconnected successfully'));
    }
}

// Initialize the plugin
PhotoWeb_Plugin::get_instance();

