<?php
/**
 * PhotoWeb Shortcode Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class PhotoWeb_Shortcode {
    
    public function __construct() {
        add_shortcode('foldious_image', array($this, 'render_image_shortcode'));
        add_shortcode('foldious_gallery', array($this, 'render_gallery_shortcode'));
    }
    
    /**
     * Render single image shortcode
     * Usage: [foldious_image code="image-code" size="full|medium|thumb" width="" height="" class="" alt=""]
     */
    public function render_image_shortcode($atts) {
        $atts = shortcode_atts(array(
            'code' => '',
            'size' => 'full',
            'width' => '',
            'height' => '',
            'class' => '',
            'alt' => '',
            'link' => 'false',
            'target' => '_self'
        ), $atts, 'foldious_image');
        
        if (empty($atts['code'])) {
            return '<p style="color: red;">[foldious_image] Error: Image code is required.</p>';
        }
        
        // Build image URL
        $image_url = $this->get_image_url($atts['code'], $atts['size']);
        
        // Build style attributes
        $style = '';
        if (!empty($atts['width'])) {
            $style .= 'width: ' . esc_attr($atts['width']) . '; ';
        }
        if (!empty($atts['height'])) {
            $style .= 'height: ' . esc_attr($atts['height']) . '; ';
        }
        
        // Build image tag
        $img_tag = '<img src="' . esc_url($image_url) . '"';
        if (!empty($atts['alt'])) {
            $img_tag .= ' alt="' . esc_attr($atts['alt']) . '"';
        }
        if (!empty($atts['class'])) {
            $img_tag .= ' class="' . esc_attr($atts['class']) . '"';
        }
        if (!empty($style)) {
            $img_tag .= ' style="' . esc_attr(trim($style)) . '"';
        }
        $img_tag .= ' />';
        
        // Wrap in link if requested
        if ($atts['link'] === 'true' || $atts['link'] === '1') {
            $full_url = $this->get_image_url($atts['code'], 'full');
            $img_tag = '<a href="' . esc_url($full_url) . '" target="' . esc_attr($atts['target']) . '">' . $img_tag . '</a>';
        }
        
        return $img_tag;
    }
    
    /**
     * Render gallery shortcode
     * Usage: [foldious_gallery folder="" limit="" columns="" size=""]
     */
    public function render_gallery_shortcode($atts) {
        $atts = shortcode_atts(array(
            'folder' => '',
            'limit' => 12,
            'columns' => 3,
            'size' => 'medium',
            'class' => ''
        ), $atts, 'foldious_gallery');
        
        $user_id = get_option('photoweb_user_id', '');
        if (empty($user_id)) {
            return '<p style="color: red;">[foldious_gallery] Error: Foldious account not connected.</p>';
        }
        
        // Get images from API
        $api = new PhotoWeb_API();
        $folder_id = !empty($atts['folder']) ? $this->get_folder_id_by_name($atts['folder'], $user_id) : null;
        $result = $api->get_images(1, $folder_id);
        
        if (!$result['success'] || empty($result['images'])) {
            return '<p>No images found.</p>';
        }
        
        $images = array_slice($result['images'], 0, intval($atts['limit']));
        $columns = intval($atts['columns']);
        
        // Build gallery HTML
        $gallery_class = 'photoweb-gallery-shortcode';
        if (!empty($atts['class'])) {
            $gallery_class .= ' ' . esc_attr($atts['class']);
        }
        
        $html = '<div class="' . $gallery_class . '" style="display: grid; grid-template-columns: repeat(' . $columns . ', 1fr); gap: 20px;">';
        
        foreach ($images as $image) {
            $image_url = $this->get_image_url($image['image_code'], $atts['size']);
            $full_url = $this->get_image_url($image['image_code'], 'full');
            
            $html .= '<div class="photoweb-gallery-shortcode-item">';
            $html .= '<a href="' . esc_url($full_url) . '" target="_blank">';
            $html .= '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image['image_name']) . '" />';
            $html .= '</a>';
            $html .= '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Get image URL from code
     */
    private function get_image_url($code, $size = 'full') {
        $api_url = get_option('photoweb_api_url', PHOTOWEB_API_BASE_URL);
        $base_url = str_replace('/api/plugin-api', '', $api_url);
        
        // Use drive.foldious.com for image proxy
        $host = 'drive.foldious.com';
        if (strpos($base_url, 'foldious.com') !== false) {
            $host = 'drive.foldious.com';
        }
        
        return 'https://' . $host . '/images.php?code=' . urlencode($code) . '&size=' . urlencode($size);
    }
    
    /**
     * Get folder ID by name
     */
    private function get_folder_id_by_name($folder_name, $user_id) {
        $api = new PhotoWeb_API();
        $result = $api->get_folders();
        
        if ($result['success'] && !empty($result['folders'])) {
            foreach ($result['folders'] as $folder) {
                if ($folder['folder_name'] === $folder_name) {
                    return $folder['folder_id'];
                }
            }
        }
        
        return null;
    }
}

new PhotoWeb_Shortcode();

