<?php
/**
 * PhotoWeb Gutenberg Block
 */

if (!defined('ABSPATH')) {
    exit;
}

class PhotoWeb_Gutenberg {
    
    public function __construct() {
        add_action('init', array($this, 'register_block'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_assets'));
    }
    
    /**
     * Register Gutenberg block
     */
    public function register_block() {
        if (!function_exists('register_block_type')) {
            return;
        }
        
        register_block_type('photoweb/foldious-image', array(
            'render_callback' => array($this, 'render_block'),
            'attributes' => array(
                'imageCode' => array(
                    'type' => 'string',
                    'default' => ''
                ),
                'size' => array(
                    'type' => 'string',
                    'default' => 'full'
                ),
                'width' => array(
                    'type' => 'string',
                    'default' => ''
                ),
                'height' => array(
                    'type' => 'string',
                    'default' => ''
                ),
                'alt' => array(
                    'type' => 'string',
                    'default' => ''
                ),
                'linkToFull' => array(
                    'type' => 'boolean',
                    'default' => false
                )
            )
        ));
    }
    
    /**
     * Enqueue block editor assets
     */
    public function enqueue_block_assets() {
        wp_enqueue_script(
            'photoweb-gutenberg-block',
            PHOTOWEB_PLUGIN_URL . 'assets/js/gutenberg-block.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
            PHOTOWEB_PLUGIN_VERSION,
            true
        );
        
        wp_enqueue_style(
            'photoweb-gutenberg-block',
            PHOTOWEB_PLUGIN_URL . 'assets/css/gutenberg-block.css',
            array('wp-edit-blocks'),
            PHOTOWEB_PLUGIN_VERSION
        );
        
        // Foldious tab feature removed from media uploader
        
        wp_localize_script('photoweb-gutenberg-block', 'photowebBlockData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('photoweb_nonce'),
            'apiUrl' => get_option('photoweb_api_url', PHOTOWEB_API_BASE_URL),
            'userId' => get_option('photoweb_user_id', '')
        ));
    }
    
    /**
     * Render block on frontend
     */
    public function render_block($attributes) {
        if (empty($attributes['imageCode'])) {
            return '';
        }
        
        $code = sanitize_text_field($attributes['imageCode']);
        $size = isset($attributes['size']) ? sanitize_text_field($attributes['size']) : 'full';
        $alt = isset($attributes['alt']) ? esc_attr($attributes['alt']) : '';
        $width = isset($attributes['width']) ? esc_attr($attributes['width']) : '';
        $height = isset($attributes['height']) ? esc_attr($attributes['height']) : '';
        $link_to_full = isset($attributes['linkToFull']) && $attributes['linkToFull'];
        
        // Build image URL
        $api_url = get_option('photoweb_api_url', PHOTOWEB_API_BASE_URL);
        $base_url = str_replace('/api/plugin-api', '', $api_url);
        $host = 'drive.foldious.com';
        $image_url = 'https://' . $host . '/images.php?code=' . urlencode($code) . '&size=' . urlencode($size);
        $full_url = 'https://' . $host . '/images.php?code=' . urlencode($code) . '&size=full';
        
        // Build style
        $style = '';
        if ($width) $style .= 'width: ' . $width . '; ';
        if ($height) $style .= 'height: ' . $height . '; ';
        
        $img_tag = '<img src="' . esc_url($image_url) . '"';
        if ($alt) $img_tag .= ' alt="' . $alt . '"';
        if ($style) $img_tag .= ' style="' . esc_attr(trim($style)) . '"';
        $img_tag .= ' />';
        
        if ($link_to_full) {
            $img_tag = '<a href="' . esc_url($full_url) . '" target="_blank">' . $img_tag . '</a>';
        }
        
        return $img_tag;
    }
}

new PhotoWeb_Gutenberg();

