<?php
/**
 * PhotoWeb Auto-Upload from Media Library
 */

if (!defined('ABSPATH')) {
    exit;
}

class PhotoWeb_Auto_Upload {
    
    public function __construct() {
        // Hook into WordPress media upload
        add_action('add_attachment', array($this, 'auto_upload_to_foldious'), 10, 1);
        
        // Add settings option
        add_action('admin_init', array($this, 'register_auto_upload_setting'));
    }
    
    /**
     * Register auto-upload setting
     */
    public function register_auto_upload_setting() {
        register_setting('photoweb_settings', 'photoweb_auto_upload', array(
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean'
        ));
    }
    
    /**
     * Auto-upload image to Foldious when added to WordPress media library
     */
    public function auto_upload_to_foldious($attachment_id) {
        // Check if auto-upload is enabled
        $auto_upload = get_option('photoweb_auto_upload', false);
        if (!$auto_upload) {
            return;
        }
        
        // Check if user is connected
        $user_id = get_option('photoweb_user_id', '');
        if (empty($user_id)) {
            return;
        }
        
        // Get attachment file
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return;
        }
        
        // Check if it's an image
        $mime_type = get_post_mime_type($attachment_id);
        if (strpos($mime_type, 'image/') !== 0) {
            return;
        }
        
        // Check if already uploaded to Foldious (prevent duplicates)
        $foldious_image_id = get_post_meta($attachment_id, '_foldious_image_id', true);
        if (!empty($foldious_image_id)) {
            return; // Already uploaded
        }
        
        // Prepare file for upload
        $file_name = basename($file_path);
        $file_type = $mime_type;
        
        // Create file array for API
        // Note: We need to copy the file to a temp location since WordPress might delete it
        $temp_file = wp_tempnam('foldious_');
        if (!copy($file_path, $temp_file)) {
            return; // Failed to copy
        }
        
        $files = array(array(
            'name' => $file_name,
            'type' => $file_type,
            'tmp_name' => $temp_file,
            'error' => 0,
            'size' => filesize($temp_file)
        ));
        
        // Upload to Foldious
        $api = new PhotoWeb_API();
        $result = $api->upload_images($files);
        
        if ($result['success'] && !empty($result['data']['image_id'])) {
            // Store Foldious image ID in attachment meta
            update_post_meta($attachment_id, '_foldious_image_id', $result['data']['image_id']);
            update_post_meta($attachment_id, '_foldious_image_code', $result['data']['image_code']);
            update_post_meta($attachment_id, '_foldious_uploaded', current_time('mysql'));
        }
    }
}

new PhotoWeb_Auto_Upload();

