<?php
/**
 * PhotoWeb API Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class PhotoWeb_API {
    
    private $api_url;
    private $user_id;
    
    public function __construct() {
        $this->api_url = get_option('photoweb_api_url', PHOTOWEB_API_BASE_URL);
        $this->user_id = get_option('photoweb_user_id', '');
    }
    
    /**
     * Validate user email
     */
    public function validate_user($email) {
        $url = $this->api_url . '/validate_user.php';
        
        $response = wp_remote_post($url, array(
            'body' => array(
                'email' => $email
            ),
            'timeout' => 30,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Log for debugging (remove in production)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('PhotoWeb API Response - URL: ' . $url . ', HTTP Code: ' . $http_code . ', Body: ' . substr($body, 0, 500));
        }
        
        // Check if response is valid JSON
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg() . ' | Response: ' . substr($body, 0, 200)
            );
        }
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Empty response from server. HTTP Code: ' . $http_code
        );
    }
    
    /**
     * Upload multiple images
     */
    public function upload_images($files, $folder_id = null) {
        // Check if user_id is set
        if (empty($this->user_id)) {
            return array(
                'success' => false,
                'message' => 'User ID is not set. Please connect your account first.'
            );
        }
        
        $url = $this->api_url . '/upload_image.php';
        
        // Build multipart form data manually for WordPress
        $boundary = wp_generate_password(12, false);
        $body = '';
        
        // Add user_id
        $body .= '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="user_id"' . "\r\n\r\n";
        $body .= (string)$this->user_id . "\r\n";
        
        // Add folder_id if provided
        if ($folder_id) {
            $body .= '--' . $boundary . "\r\n";
            $body .= 'Content-Disposition: form-data; name="folder_id"' . "\r\n\r\n";
            $body .= (string)$folder_id . "\r\n";
        }
        
        // Add all image files - upload_image.php expects 'images[]' array
        foreach ($files as $file) {
            $file_path = $file['tmp_name'];
            $file_name = $file['name'];
            $file_type = $file['type'];
            
            $body .= '--' . $boundary . "\r\n";
            $body .= 'Content-Disposition: form-data; name="images[]"; filename="' . esc_attr($file_name) . '"' . "\r\n";
            $body .= 'Content-Type: ' . $file_type . "\r\n\r\n";
            $body .= file_get_contents($file_path) . "\r\n";
        }
        
        $body .= '--' . $boundary . '--';
        
        // Use wp_remote_post with manually constructed multipart body
        $response = wp_remote_post($url, array(
            'body' => $body,
            'headers' => array(
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary
            ),
            'timeout' => 120,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        // Check if response is valid JSON
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg(),
                'http_code' => $http_code,
                'response_preview' => substr($response_body, 0, 200)
            );
        }
        
        if ($http_code === 200 && $data) {
            // Check if plugin API returned success
            if (isset($data['success']) && $data['success'] === true) {
                return $data; // Plugin API already formatted correctly
            }
            
            // Check if main API returned success (direct call)
            if (isset($data['status']) && $data['status'] === 'success') {
                // Main API returns uploaded_images array
                return array(
                    'success' => true,
                    'data' => array(
                        'uploaded_images' => $data['uploaded_images'] ?? array(),
                        'uploaded_count' => $data['uploaded_count'] ?? 0,
                        'failed_count' => $data['failed_count'] ?? 0,
                        'total_images' => $data['total_images'] ?? 0
                    )
                );
            }
        }
        
        // Handle error response
        $error_msg = 'Upload failed';
        if ($data && isset($data['message'])) {
            $error_msg = $data['message'];
        } elseif ($data && isset($data['status']) && $data['status'] === 'error') {
            $error_msg = $data['message'] ?? 'Upload failed';
        }
        
        return array(
            'success' => false,
            'message' => $error_msg,
            'http_code' => $http_code,
            'debug' => $data
        );
    }
    
    /**
     * Upload single image (backward compatibility)
     */
    public function upload_image($file, $folder_id = null) {
        return $this->upload_images(array($file), $folder_id);
        // Check if user_id is set
        if (empty($this->user_id)) {
            return array(
                'success' => false,
                'message' => 'User ID is not set. Please connect your account first.'
            );
        }
        
        $url = $this->api_url . '/upload_image.php';
        
        // Build multipart form data manually for WordPress
        $boundary = wp_generate_password(12, false);
        $body = '';
        
        // Add user_id
        $body .= '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="user_id"' . "\r\n\r\n";
        $body .= (string)$this->user_id . "\r\n";
        
        // Add folder_id if provided
        if ($folder_id) {
            $body .= '--' . $boundary . "\r\n";
            $body .= 'Content-Disposition: form-data; name="folder_id"' . "\r\n\r\n";
            $body .= (string)$folder_id . "\r\n";
        }
        
        // Add image file - upload_image.php expects 'images' (plural)
        $file_path = $file['tmp_name'];
        $file_name = $file['name'];
        $file_type = $file['type'];
        
        $body .= '--' . $boundary . "\r\n";
        $body .= 'Content-Disposition: form-data; name="images"; filename="' . esc_attr($file_name) . '"' . "\r\n";
        $body .= 'Content-Type: ' . $file_type . "\r\n\r\n";
        $body .= file_get_contents($file_path) . "\r\n";
        $body .= '--' . $boundary . '--';
        
        // Use wp_remote_post with manually constructed multipart body
        $response = wp_remote_post($url, array(
            'body' => $body,
            'headers' => array(
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary
            ),
            'timeout' => 120,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        // Check if response is valid JSON
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg(),
                'http_code' => $http_code,
                'response_preview' => substr($response_body, 0, 200)
            );
        }
        
        if ($http_code === 200 && $data) {
            // Check if plugin API returned success
            if (isset($data['success']) && $data['success'] === true) {
                return $data; // Plugin API already formatted correctly
            }
            
            // Check if main API returned success (direct call)
            if (isset($data['status']) && $data['status'] === 'success') {
                // Main API returns uploaded_images array directly
                if (isset($data['uploaded_images']) && is_array($data['uploaded_images']) && count($data['uploaded_images']) > 0) {
                    // Return the first uploaded image
                    return array(
                        'success' => true,
                        'data' => $data['uploaded_images'][0]
                    );
                } else {
                    // No images returned
                    return array(
                        'success' => false,
                        'message' => 'Upload completed but no images returned',
                        'uploaded_count' => isset($data['uploaded_count']) ? $data['uploaded_count'] : 0,
                        'failed_count' => isset($data['failed_count']) ? $data['failed_count'] : 0
                    );
                }
            }
        }
        
        // Handle error response
        $error_msg = 'Upload failed';
        if ($data && isset($data['message'])) {
            $error_msg = $data['message'];
        } elseif ($data && isset($data['status']) && $data['status'] === 'error') {
            $error_msg = $data['message'] ?? 'Upload failed';
        }
        
        return array(
            'success' => false,
            'message' => $error_msg,
            'http_code' => $http_code,
            'debug' => $data
        );
    }
    
    /**
     * Get images
     */
    public function get_images($page = 1, $folder_id = null) {
        $url = $this->api_url . '/get_images.php';
        
        $args = array(
            'user_id' => $this->user_id,
            'page' => $page,
            'limit' => 20
        );
        
        if ($folder_id) {
            $args['folder_id'] = $folder_id;
        }
        
        $response = wp_remote_get(add_query_arg($args, $url), array(
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Invalid response from server'
        );
    }
    
    /**
     * Delete image
     */
    public function delete_image($image_id) {
        $url = $this->api_url . '/delete_image.php';
        
        $response = wp_remote_post($url, array(
            'body' => array(
                'user_id' => $this->user_id,
                'image_id' => $image_id
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Invalid response from server'
        );
    }
    
    /**
     * Get folders
     */
    public function get_folders() {
        $url = $this->api_url . '/get_folders.php';
        
        $response = wp_remote_get(add_query_arg(array('user_id' => $this->user_id), $url), array(
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Invalid response from server'
        );
    }
    
    /**
     * Get user details including default folder
     */
    public function get_user_details() {
        $url = $this->api_url . '/get_user_details.php';
        
        $response = wp_remote_get(add_query_arg(array('user_id' => $this->user_id), $url), array(
            'timeout' => 30,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg()
            );
        }
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Empty response from server. HTTP Code: ' . $http_code
        );
    }
    
    /**
     * Get user statistics
     */
    public function get_user_stats() {
        $url = $this->api_url . '/get_user_stats.php';
        
        $response = wp_remote_get(add_query_arg(array('user_id' => $this->user_id), $url), array(
            'timeout' => 30,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Debug: Log the raw response for troubleshooting
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('PhotoWeb API Response (get_user_stats): ' . substr($body, 0, 500));
        }
        
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg(),
                'http_code' => $http_code,
                'response_preview' => substr($body, 0, 200)
            );
        }
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Empty response from server. HTTP Code: ' . $http_code
        );
    }
    
    /**
     * Get image details
     */
    public function get_image_details($image_id) {
        $url = $this->api_url . '/get_image_details.php';
        
        $response = wp_remote_get(add_query_arg(array('image_id' => $image_id, 'user_id' => $this->user_id), $url), array(
            'timeout' => 30,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'success' => false,
                'message' => 'Invalid JSON response: ' . json_last_error_msg()
            );
        }
        
        return $data ? $data : array(
            'success' => false,
            'message' => 'Empty response from server. HTTP Code: ' . $http_code
        );
    }
}
