(function(blocks, element, components, editor, i18n) {
    const { registerBlockType } = blocks;
    const { createElement: el } = element;
    const { TextControl, SelectControl, ToggleControl, PanelBody, Placeholder, Button } = components;
    const { InspectorControls, MediaPlaceholder } = editor;
    const { __ } = i18n;
    
    registerBlockType('photoweb/foldious-image', {
        title: __('Foldious Image', 'photoweb'),
        icon: 'format-image',
        category: 'media',
        attributes: {
            imageCode: {
                type: 'string',
                default: ''
            },
            size: {
                type: 'string',
                default: 'full'
            },
            width: {
                type: 'string',
                default: ''
            },
            height: {
                type: 'string',
                default: ''
            },
            alt: {
                type: 'string',
                default: ''
            },
            linkToFull: {
                type: 'boolean',
                default: false
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { imageCode, size, width, height, alt, linkToFull } = attributes;
            
            // Check if user is connected
            const userId = photowebBlockData.userId;
            if (!userId) {
                return el(Placeholder, {
                    label: __('Foldious Image', 'photoweb'),
                    instructions: __('Please connect your Foldious account in the plugin settings first.', 'photoweb')
                });
            }
            
            // If no image code, show image selector
            if (!imageCode) {
                return el('div', { className: 'photoweb-gutenberg-placeholder' },
                    el(Placeholder, {
                        label: __('Select Foldious Image', 'photoweb'),
                        instructions: __('Click to browse your Foldious images', 'photoweb')
                    },
                    el(Button, {
                        isPrimary: true,
                        onClick: function() {
                            // Open media library with Foldious tab
                            if (wp && wp.media) {
                                const frame = wp.media({
                                    title: __('Select Foldious Image', 'photoweb'),
                                    button: {
                                        text: __('Select Image', 'photoweb')
                                    },
                                    library: {
                                        type: 'image'
                                    }
                                });
                                
                                frame.on('select', function() {
                                    const attachment = frame.state().get('selection').first().toJSON();
                                    // This would need custom implementation to get Foldious image code
                                    // For now, we'll use a simple input
                                });
                                
                                frame.open();
                            }
                        }
                    }, __('Browse Images', 'photoweb'))
                ));
            }
            
            // Build image URL for preview
            const imageUrl = 'https://drive.foldious.com/images.php?code=' + encodeURIComponent(imageCode) + '&size=' + encodeURIComponent(size);
            
            return el('div', { className: 'photoweb-gutenberg-image-block' },
                el(InspectorControls, {},
                    el(PanelBody, { title: __('Image Settings', 'photoweb') },
                        el(TextControl, {
                            label: __('Image Code', 'photoweb'),
                            value: imageCode,
                            onChange: function(value) {
                                setAttributes({ imageCode: value });
                            }
                        }),
                        el(SelectControl, {
                            label: __('Size', 'photoweb'),
                            value: size,
                            options: [
                                { label: __('Full', 'photoweb'), value: 'full' },
                                { label: __('Medium', 'photoweb'), value: 'medium' },
                                { label: __('Thumbnail', 'photoweb'), value: 'thumb' }
                            ],
                            onChange: function(value) {
                                setAttributes({ size: value });
                            }
                        }),
                        el(TextControl, {
                            label: __('Width', 'photoweb'),
                            value: width,
                            placeholder: __('e.g., 500px', 'photoweb'),
                            onChange: function(value) {
                                setAttributes({ width: value });
                            }
                        }),
                        el(TextControl, {
                            label: __('Height', 'photoweb'),
                            value: height,
                            placeholder: __('e.g., 300px', 'photoweb'),
                            onChange: function(value) {
                                setAttributes({ height: value });
                            }
                        }),
                        el(TextControl, {
                            label: __('Alt Text', 'photoweb'),
                            value: alt,
                            onChange: function(value) {
                                setAttributes({ alt: value });
                            }
                        }),
                        el(ToggleControl, {
                            label: __('Link to Full Image', 'photoweb'),
                            checked: linkToFull,
                            onChange: function(value) {
                                setAttributes({ linkToFull: value });
                            }
                        })
                    )
                ),
                el('div', { className: 'photoweb-gutenberg-image-preview' },
                    el('img', {
                        src: imageUrl,
                        alt: alt || __('Foldious Image', 'photoweb'),
                        style: {
                            maxWidth: '100%',
                            height: 'auto'
                        }
                    }),
                    el('p', { style: { marginTop: '10px', fontSize: '12px', color: '#666' } },
                        __('Image Code: ', 'photoweb') + imageCode
                    )
                )
            );
        },
        save: function() {
            // Server-side rendering
            return null;
        }
    });
})(
    window.wp.blocks,
    window.wp.element,
    window.wp.components,
    window.wp.editor,
    window.wp.i18n
);

