/**
 * PhotoWeb WordPress Plugin Admin JavaScript
 */

(function($) {
    'use strict';
    
    // Setup Form
    $('#photoweb-setup-form').on('submit', function(e) {
        e.preventDefault();
        
        const email = $('#user-email').val();
        const $message = $('#photoweb-setup-message');
        const $submitBtn = $(this).find('button[type="submit"]');
        
        if (!email) {
            showMessage($message, 'Please enter your email address', 'error');
            return;
        }
        
        $submitBtn.prop('disabled', true).text('Validating...');
        $message.hide();
        
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_validate_user',
                nonce: photowebData.nonce,
                email: email
            },
            success: function(response) {
                if (response.success) {
                    showToast('success', 'Account connected successfully! Redirecting...');
                    showMessage($message, 'Account connected successfully! Redirecting...', 'success');
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                } else {
                    // Check if user is not registered
                    if (response.data && response.data.not_registered) {
                        const errorMessage = 'User not found or inactive.<br><strong>Create account on <a href="https://foldious.com" target="_blank" rel="noopener noreferrer">foldious.com</a> to get access.</strong>';
                        showMessage($message, errorMessage, 'error', true);
                    } else {
                        showMessage($message, response.data.message || 'Failed to validate user', 'error');
                    }
                    $submitBtn.prop('disabled', false).html('<span class="dashicons dashicons-yes"></span> Validate & Connect');
                }
            },
            error: function() {
                showMessage($message, 'An error occurred. Please try again.', 'error');
                $submitBtn.prop('disabled', false).html('<span class="dashicons dashicons-yes"></span> Validate & Connect');
            }
        });
    });
    
    // Upload Form
    const $uploadArea = $('#photoweb-upload-area');
    const $fileInput = $('#upload-images');
    const $preview = $('#photoweb-upload-preview');
    let selectedFiles = [];
    let failedUploads = []; // Store failed uploads for retry
    let uploadQueue = []; // Store upload queue
    
    // Click to select files - use label approach to avoid infinite loop
    // The upload area is now a label that wraps the file input
    
    // Drag and drop
    $uploadArea.on('dragover', function(e) {
        e.preventDefault();
        $(this).addClass('dragover');
    });
    
    $uploadArea.on('dragleave', function() {
        $(this).removeClass('dragover');
    });
    
    $uploadArea.on('drop', function(e) {
        e.preventDefault();
        $(this).removeClass('dragover');
        
        const files = e.originalEvent.dataTransfer.files;
        handleFiles(files);
    });
    
    // File input change
    $fileInput.on('change', function() {
        handleFiles(this.files);
    });
    
    function handleFiles(files) {
        const validFiles = [];
        const maxSize = 10 * 1024 * 1024; // 10MB
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        const maxFiles = 50;
        
        // Check total file count
        if (selectedFiles.length + files.length > maxFiles) {
            showToast('error', `Maximum ${maxFiles} files allowed. Please select fewer files.`);
            return;
        }
        
        Array.from(files).forEach((file) => {
            // Validate file type
            if (!allowedTypes.includes(file.type)) {
                showToast('error', `"${file.name}" is not a valid image file. Allowed types: JPG, PNG, GIF, WebP`);
                return; // Skip this file, continue with next
            }
            
            // Validate file size
            if (file.size > maxSize) {
                showToast('error', `"${file.name}" is too large (${formatFileSize(file.size)}). Maximum size is 10MB.`);
                return; // Skip this file, continue with next
            }
            
            // Check for duplicates
            const isDuplicate = selectedFiles.some(existingFile => 
                existingFile.name === file.name && existingFile.size === file.size
            );
            
            if (isDuplicate) {
                showToast('warning', `"${file.name}" is already selected. Skipping duplicate.`);
                return; // Skip this file, continue with next
            }
            
            validFiles.push(file);
        });
        
        if (validFiles.length > 0) {
            selectedFiles = selectedFiles.concat(validFiles);
            displayPreview();
            showToast('success', `${validFiles.length} file(s) added successfully.`);
        }
    }
    
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }
    
    function displayPreview() {
        $preview.empty().show();
        
        selectedFiles.forEach((file, index) => {
            if (!file.type.startsWith('image/')) return;
            
            const reader = new FileReader();
            reader.onload = function(e) {
                const $item = $('<div class="photoweb-upload-preview-item" data-file-index="' + index + '">' +
                    '<img src="' + e.target.result + '" alt="' + file.name + '">' +
                    '<div class="photoweb-preview-overlay">' +
                    '<span class="photoweb-preview-name">' + escapeHtml(file.name) + '</span>' +
                    '<span class="photoweb-preview-size">' + formatFileSize(file.size) + '</span>' +
                    '</div>' +
                    '<button type="button" class="remove" data-index="' + index + '">×</button>' +
                    '</div>');
                $preview.append($item);
            };
            reader.readAsDataURL(file);
        });
    }
    
    function createUploadQueueItem(file, index) {
        const fileName = escapeHtml(file.name);
        const fileSize = formatFileSize(file.size);
        const $queueItem = $('<div class="photoweb-upload-queue-item waiting" data-file-index="' + index + '">' +
            '<div class="photoweb-queue-item-info">' +
            '<div class="photoweb-queue-item-thumb">' +
            '<img src="" alt="' + file.name + '">' +
            '</div>' +
            '<div class="photoweb-queue-item-details">' +
            '<div class="photoweb-queue-item-name">' + fileName + '</div>' +
            '<div class="photoweb-queue-item-size">' + fileSize + '</div>' +
            '</div>' +
            '</div>' +
            '<div class="photoweb-queue-item-progress">' +
            '<div class="photoweb-queue-progress-bar">' +
            '<div class="photoweb-queue-progress-fill" style="width: 0%"></div>' +
            '</div>' +
            '<div class="photoweb-queue-item-status">' +
            '<span class="photoweb-queue-status-text">Waiting...</span>' +
            '<span class="photoweb-queue-status-percent">0%</span>' +
            '</div>' +
            '</div>' +
            '</div>');
        
        // Load thumbnail
        const reader = new FileReader();
        reader.onload = function(e) {
            $queueItem.find('.photoweb-queue-item-thumb img').attr('src', e.target.result);
        };
        reader.readAsDataURL(file);
        
        return $queueItem;
    }
    
    function updateUploadQueueItem(index, progress, status) {
        const $item = $('#photoweb-upload-queue-list').find('[data-file-index="' + index + '"]');
        if ($item.length === 0) return;
        
        const $progressFill = $item.find('.photoweb-queue-progress-fill');
        const $statusText = $item.find('.photoweb-queue-status-text');
        const $statusPercent = $item.find('.photoweb-queue-status-percent');
        
        $progressFill.css('width', progress + '%');
        $statusPercent.text(Math.round(progress) + '%');
        
        if (status) {
            const statusMessages = {
                'waiting': 'Waiting...',
                'compressing': 'Compressing...',
                'compressed': 'Compressed',
                'compression-failed': 'Compression failed',
                'uploading': 'Uploading...',
                'success': 'Uploaded',
                'error': 'Failed'
            };
            
            $statusText.text(statusMessages[status] || status);
            $item.removeClass('waiting compressing compressed compression-failed uploading success error');
            $item.addClass(status);
        }
    }
    
    function initializeUploadQueue() {
        const $queue = $('#photoweb-upload-queue');
        const $queueList = $('#photoweb-upload-queue-list');
        
        $queueList.empty();
        $queue.hide();
        
        if (selectedFiles.length === 0) return;
        
        $queue.show();
        
        selectedFiles.forEach((file, index) => {
            const $queueItem = createUploadQueueItem(file, index);
            $queueList.append($queueItem);
        });
    }
    
    // Remove preview item
    $preview.on('click', '.remove', function() {
        const index = $(this).data('index');
        selectedFiles.splice(index, 1);
        displayPreview();
    });
    
    // Retry failed uploads
    function showRetryOption() {
        if (failedUploads.length > 0) {
            $('#photoweb-retry-count').text(failedUploads.length);
            $('#photoweb-upload-retry').show();
        } else {
            $('#photoweb-upload-retry').hide();
        }
    }
    
    $('#photoweb-retry-btn').on('click', function() {
        if (failedUploads.length === 0) {
            showToast('info', 'No failed uploads to retry');
            return;
        }
        
        // Add failed files back to selected files
        selectedFiles = selectedFiles.concat(failedUploads);
        failedUploads = [];
        
        // Update preview
        displayPreview();
        
        // Hide retry section
        $('#photoweb-upload-retry').hide();
        
        // Trigger upload
        $('#photoweb-upload-form').submit();
    });
    
    $('#photoweb-clear-failed-btn').on('click', function() {
        if (confirm('Are you sure you want to clear all failed uploads?')) {
            failedUploads = [];
            $('#photoweb-upload-retry').hide();
            showToast('info', 'Failed uploads cleared');
        }
    });
    
    // Compression settings toggle
    $('#photoweb-enable-compression').on('change', function() {
        if ($(this).is(':checked')) {
            $('#photoweb-compression-settings').slideDown();
        } else {
            $('#photoweb-compression-settings').slideUp();
        }
    });
    
    // Update quality value display
    $('#photoweb-compression-quality').on('input', function() {
        $('#photoweb-quality-value').text($(this).val());
    });
    
    // Compress image function
    function compressImage(file, quality = 0.85) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = function(e) {
                const img = new Image();
                img.onload = function() {
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');
                    
                    // Calculate new dimensions (maintain aspect ratio, max 1920px)
                    let width = img.width;
                    let height = img.height;
                    const maxDimension = 1920;
                    
                    if (width > maxDimension || height > maxDimension) {
                        if (width > height) {
                            height = (height / width) * maxDimension;
                            width = maxDimension;
                        } else {
                            width = (width / height) * maxDimension;
                            height = maxDimension;
                        }
                    }
                    
                    canvas.width = width;
                    canvas.height = height;
                    
                    // Draw and compress
                    ctx.drawImage(img, 0, 0, width, height);
                    
                    canvas.toBlob(function(blob) {
                        if (blob) {
                            // Create a new File object with compressed data
                            const compressedFile = new File([blob], file.name, {
                                type: file.type,
                                lastModified: Date.now()
                            });
                            resolve(compressedFile);
                        } else {
                            reject(new Error('Compression failed'));
                        }
                    }, file.type, quality);
                };
                img.onerror = reject;
                img.src = e.target.result;
            };
            reader.onerror = reject;
            reader.readAsDataURL(file);
        });
    }
    
    // Upload form submit
    $('#photoweb-upload-form').on('submit', function(e) {
        e.preventDefault();
        
        if (selectedFiles.length === 0) {
            showToast('error', 'Please select at least one image');
            return;
        }
        
        const $submitBtn = $('#photoweb-upload-btn');
        const $progress = $('#photoweb-upload-progress');
        const $progressFill = $('#photoweb-progress-fill');
        const $progressText = $('#photoweb-progress-text');
        const $message = $('#photoweb-upload-message');
        
        const enableCompression = $('#photoweb-enable-compression').is(':checked');
        const compressionQuality = enableCompression ? parseFloat($('#photoweb-compression-quality').val()) / 100 : 1.0;
        
        $submitBtn.prop('disabled', true).text('Preparing...');
        $progress.show();
        $message.hide();
        
        // Initialize upload queue
        initializeUploadQueue();
        
        // Process files (compress if enabled)
        let filesToUpload = selectedFiles;
        
        if (enableCompression) {
            $progressText.text('Compressing images...');
            $progressFill.css('width', '10%');
            
            const compressionPromises = selectedFiles.map((file, index) => {
                return compressImage(file, compressionQuality).then(compressedFile => {
                    const progress = 10 + ((index + 1) / selectedFiles.length) * 20;
                    $progressFill.css('width', progress + '%');
                    $progressText.text(`Compressing: ${index + 1}/${selectedFiles.length}`);
                    return compressedFile;
                }).catch(error => {
                    console.warn('Compression failed for', file.name, error);
                    return file; // Use original if compression fails
                });
            });
            
            Promise.all(compressionPromises).then(compressedFiles => {
                filesToUpload = compressedFiles;
                startUpload(filesToUpload);
            });
        } else {
            startUpload(filesToUpload);
        }
        
        function startUpload(files) {
            // Get folder ID
            let folderId = $('#upload-folder').val();
            if (!folderId && photowebData.defaultFolderId) {
                folderId = photowebData.defaultFolderId;
            }
            
            // Upload files one by one to show individual progress
            let uploadedCount = 0;
            let failedCount = 0;
            const totalFiles = files.length;
            
            // Update overall progress
            $progressText.text('Uploading: 0/' + totalFiles);
            $progressFill.css('width', enableCompression ? '30%' : '0%');
            
            // Upload each file individually
            const uploadPromises = files.map((file, index) => {
                return uploadSingleFile(file, index, folderId).then(function(result) {
                    if (result.success) {
                        uploadedCount++;
                        updateUploadQueueItem(index, 100, 'success');
                    } else {
                        failedCount++;
                        updateUploadQueueItem(index, 0, 'error');
                    }
                    
                    // Update overall progress
                    const overallProgress = enableCompression ? 30 : 0;
                    const uploadProgress = ((uploadedCount + failedCount) / totalFiles) * 70;
                    $progressFill.css('width', (overallProgress + uploadProgress) + '%');
                    $progressText.text(`Uploading: ${uploadedCount + failedCount}/${totalFiles}`);
                    
                    return result;
                });
            });
            
            Promise.all(uploadPromises).then(function(results) {
                handleUploadComplete(results, uploadedCount, failedCount, totalFiles);
            });
        }
        
        function uploadSingleFile(file, index, folderId) {
            return new Promise(function(resolve) {
                updateUploadQueueItem(index, 0, 'uploading');
                
                const formData = new FormData();
                formData.append('action', 'photoweb_upload_image');
                formData.append('nonce', photowebData.nonce);
                formData.append('folder_id', folderId || '');
                formData.append('images[]', file);
                
                $.ajax({
                    url: photowebData.ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    xhr: function() {
                        const xhr = new window.XMLHttpRequest();
                        xhr.upload.addEventListener('progress', function(e) {
                            if (e.lengthComputable) {
                                const percent = (e.loaded / e.total) * 100;
                                updateUploadQueueItem(index, percent, 'uploading');
                            }
                        });
                        return xhr;
                    },
                    success: function(response) {
                        if (response.success) {
                            resolve({ success: true, response: response });
                        } else {
                            resolve({ success: false, error: response.data.message || 'Upload failed' });
                        }
                    },
                    error: function(xhr, status, error) {
                        resolve({ success: false, error: error || 'Network error' });
                    }
                });
            });
        }
        
        function handleUploadComplete(results, uploadedCount, failedCount, totalFiles) {
            $progressFill.css('width', '100%');
            $progressText.text('Upload complete');
            
            // Check if there were any failures
            if (failedCount > 0) {
                const errorMessage = `${uploadedCount} uploaded successfully, ${failedCount} failed.`;
                showToast('warning', errorMessage, 6000);
                
                // Store failed files for retry
                failedUploads = selectedFiles.slice();
                selectedFiles = [];
                showRetryOption();
            } else {
                showToast('success', `${uploadedCount} image(s) uploaded successfully!`);
                selectedFiles = [];
                failedUploads = [];
                $('#photoweb-upload-retry').hide();
            }
            
            $fileInput.val('');
            $preview.hide().empty();
            
            if (failedUploads.length === 0) {
                setTimeout(function() {
                    window.location.href = '?page=photoweb-gallery';
                }, 2000);
            } else {
                $submitBtn.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> Upload Images');
            }
            
            // Hide progress after delay
            setTimeout(function() {
                $progress.hide();
            }, 3000);
        }
    });
    
    // Gallery
    let currentPage = 1;
    let currentFolderId = null;
    let currentSearchQuery = '';
    let currentSortBy = 'date_desc'; // Default sort: newest first
    let allImages = []; // Store all images for client-side search
    let searchTimeout = null;
    let selectedImageIds = []; // Store selected image IDs for bulk operations
    
    // Search functionality
    $('#gallery-search').on('input', function() {
        const query = $(this).val().trim().toLowerCase();
        currentSearchQuery = query;
        currentPage = 1; // Reset to first page on search
        
        // Debounce search
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(function() {
            if (query.length > 0) {
                // Client-side search if we have all images loaded
                if (allImages.length > 0) {
                    filterImagesBySearch(query);
                } else {
                    // Server-side search - reload gallery
                    loadGallery(1, currentFolderId);
                }
            } else {
                // Clear search - reload gallery
                loadGallery(1, currentFolderId);
            }
        }, 300);
    });
    
    function filterImagesBySearch(query) {
        const $grid = $('#photoweb-gallery-grid');
        const $empty = $('#photoweb-gallery-empty');
        const $pagination = $('#photoweb-gallery-pagination');
        
        let filtered = allImages.filter(function(image) {
            const name = (image.image_name || '').toLowerCase();
            return name.includes(query);
        });
        
        // Apply sort to filtered results
        if (filtered.length > 0 && currentSortBy) {
            filtered = sortImages(filtered, currentSortBy);
        }
        
        if (filtered.length === 0) {
            $grid.hide();
            $empty.show();
            $pagination.hide();
        } else {
            $grid.show();
            $empty.hide();
            displayImages(filtered);
            $pagination.hide(); // Hide pagination for search results
        }
    }
    
    function loadGallery(page = 1, folderId = null) {
        const $loader = $('#photoweb-gallery-loader');
        const $grid = $('#photoweb-gallery-grid');
        const $empty = $('#photoweb-gallery-empty');
        const $pagination = $('#photoweb-gallery-pagination');
        
        $loader.show();
        $grid.hide();
        $empty.hide();
        $pagination.hide();
        
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_get_images',
                nonce: photowebData.nonce,
                page: page,
                folder_id: folderId || ''
            },
            success: function(response) {
                $loader.hide();
                
                if (response.success && response.data.images && response.data.images.length > 0) {
                    // Store all images for search (if first page and no folder filter)
                    if (page === 1 && !folderId) {
                        allImages = response.data.images;
                    }
                    
                    // Apply sort
                    let imagesToDisplay = response.data.images;
                    if (currentSortBy && currentSortBy !== 'date_desc') {
                        imagesToDisplay = sortImages(imagesToDisplay, currentSortBy);
                    }
                    
                    // Apply search filter if active
                    if (currentSearchQuery.length > 0) {
                        filterImagesBySearch(currentSearchQuery);
                    } else {
                        displayImages(imagesToDisplay);
                        displayPagination(response.data.pagination);
                        $grid.show();
                    }
                } else {
                    $empty.show();
                }
            },
            error: function() {
                $loader.hide();
                $empty.show();
                showToast('error', 'Failed to load images. Please try again.');
            }
        });
    }
    
    function displayImages(images) {
        const $grid = $('#photoweb-gallery-grid');
        $grid.empty();
        
        images.forEach(function(image) {
            const isSelected = selectedImageIds.includes(image.image_id);
            const $item = $('<div class="photoweb-gallery-item' + (isSelected ? ' selected' : '') + '" data-image-id="' + image.image_id + '">' +
                '<div class="photoweb-gallery-item-checkbox">' +
                '<input type="checkbox" class="photoweb-image-checkbox" data-image-id="' + image.image_id + '"' + (isSelected ? ' checked' : '') + '>' +
                '</div>' +
                '<div class="photoweb-gallery-item-actions">' +
                '<button type="button" class="view-image-details" data-image-id="' + image.image_id + '" title="View Details">👁️</button>' +
                '<button type="button" class="copy-url" data-url="' + image.image_medium + '" title="Copy URL">📋</button>' +
                '<button type="button" class="delete-image" data-id="' + image.image_id + '" title="Delete">🗑️</button>' +
                '</div>' +
                '<img src="' + image.image_thumb + '" alt="' + image.image_name + '" loading="lazy" class="photoweb-gallery-image">' +
                '<div class="photoweb-gallery-item-info">' +
                '<h4>' + escapeHtml(image.image_name) + '</h4>' +
                '<p>' + image.image_size_mb + ' MB</p>' +
                '</div>' +
                '</div>');
            $grid.append($item);
        });
        
        updateBulkActionsUI();
    }
    
    function displayPagination(pagination) {
        if (pagination.total_pages <= 1) {
            $('#photoweb-gallery-pagination').hide();
            return;
        }
        
        const $pagination = $('#photoweb-gallery-pagination');
        let html = '<div class="photoweb-pagination-wrapper">';
        
        const currentPage = pagination.page;
        const totalPages = pagination.total_pages;
        const total = pagination.total;
        
        // Show total count
        html += '<div class="photoweb-pagination-info">';
        html += 'Showing ' + ((currentPage - 1) * pagination.limit + 1) + '-' + Math.min(currentPage * pagination.limit, total) + ' of ' + total + ' images';
        html += '</div>';
        
        // Pagination controls
        html += '<div class="photoweb-pagination-controls">';
        
        // First page button
        if (currentPage > 1) {
            html += '<button class="button" data-page="1" title="First page">««</button>';
        }
        
        // Previous button
        if (currentPage > 1) {
            html += '<button class="button" data-page="' + (currentPage - 1) + '" title="Previous page">‹ Previous</button>';
        }
        
        // Page numbers
        html += '<div class="photoweb-pagination-numbers">';
        
        // Calculate which page numbers to show
        let startPage = Math.max(1, currentPage - 2);
        let endPage = Math.min(totalPages, currentPage + 2);
        
        // Adjust if we're near the start
        if (currentPage <= 3) {
            endPage = Math.min(5, totalPages);
        }
        
        // Adjust if we're near the end
        if (currentPage >= totalPages - 2) {
            startPage = Math.max(1, totalPages - 4);
        }
        
        // Show first page if we're not showing it
        if (startPage > 1) {
            html += '<button class="button" data-page="1">1</button>';
            if (startPage > 2) {
                html += '<span class="photoweb-pagination-ellipsis">...</span>';
            }
        }
        
        // Show page numbers
        for (let i = startPage; i <= endPage; i++) {
            if (i === currentPage) {
                html += '<button class="button button-primary" disabled>' + i + '</button>';
            } else {
                html += '<button class="button" data-page="' + i + '">' + i + '</button>';
            }
        }
        
        // Show last page if we're not showing it
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                html += '<span class="photoweb-pagination-ellipsis">...</span>';
            }
            html += '<button class="button" data-page="' + totalPages + '">' + totalPages + '</button>';
        }
        
        html += '</div>'; // End pagination-numbers
        
        // Next button
        if (currentPage < totalPages) {
            html += '<button class="button" data-page="' + (currentPage + 1) + '" title="Next page">Next ›</button>';
        }
        
        // Last page button
        if (currentPage < totalPages) {
            html += '<button class="button" data-page="' + totalPages + '" title="Last page">»»</button>';
        }
        
        html += '</div>'; // End pagination-controls
        html += '</div>'; // End pagination-wrapper
        
        $pagination.html(html).show();
    }
    
    // Load gallery on page load
    if ($('#photoweb-gallery-grid').length) {
        loadGallery();
        loadFolders();
    }
    
    // Load folders for filter and upload dropdown
    function loadFolders() {
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_get_folders',
                nonce: photowebData.nonce
            },
            success: function(response) {
                if (response.success && response.data.folders) {
                    const $uploadFolder = $('#upload-folder');
                    const $galleryFilter = $('#gallery-folder-filter');
                    
                    // Clear existing options (except first)
                    $uploadFolder.find('option:not(:first)').remove();
                    $galleryFilter.find('option:not(:first)').remove();
                    
                    // Add folders
                    response.data.folders.forEach(function(folder) {
                        const option = '<option value="' + folder.folder_id + '">' + 
                                      escapeHtml(folder.folder_name) + 
                                      ' (' + folder.image_count + ')</option>';
                        $uploadFolder.append(option);
                        $galleryFilter.append(option);
                    });
                }
            }
        });
    }
    
    // Filter by folder
    $('#gallery-folder-filter').on('change', function() {
        currentFolderId = $(this).val() || null;
        currentPage = 1;
        loadGallery(currentPage, currentFolderId);
    });
    
    // Sort change
    $('#gallery-sort').on('change', function() {
        currentSortBy = $(this).val();
        // If we have all images loaded, sort client-side
        if (allImages.length > 0 && currentSearchQuery.length === 0) {
            sortAndDisplayImages(allImages);
        } else {
            // Reload from server with new sort
            currentPage = 1;
            loadGallery(currentPage, currentFolderId);
        }
    });
    
    // Sort images function
    function sortImages(images, sortBy) {
        const sorted = images.slice(); // Create a copy
        
        switch(sortBy) {
            case 'date_desc':
                sorted.sort((a, b) => {
                    const dateA = new Date(a.created_at || 0);
                    const dateB = new Date(b.created_at || 0);
                    return dateB - dateA; // Newest first
                });
                break;
            case 'date_asc':
                sorted.sort((a, b) => {
                    const dateA = new Date(a.created_at || 0);
                    const dateB = new Date(b.created_at || 0);
                    return dateA - dateB; // Oldest first
                });
                break;
            case 'name_asc':
                sorted.sort((a, b) => {
                    const nameA = (a.image_name || '').toLowerCase();
                    const nameB = (b.image_name || '').toLowerCase();
                    return nameA.localeCompare(nameB); // A-Z
                });
                break;
            case 'name_desc':
                sorted.sort((a, b) => {
                    const nameA = (a.image_name || '').toLowerCase();
                    const nameB = (b.image_name || '').toLowerCase();
                    return nameB.localeCompare(nameA); // Z-A
                });
                break;
            case 'size_desc':
                sorted.sort((a, b) => {
                    const sizeA = parseFloat(a.image_size_mb || 0);
                    const sizeB = parseFloat(b.image_size_mb || 0);
                    return sizeB - sizeA; // Largest first
                });
                break;
            case 'size_asc':
                sorted.sort((a, b) => {
                    const sizeA = parseFloat(a.image_size_mb || 0);
                    const sizeB = parseFloat(b.image_size_mb || 0);
                    return sizeA - sizeB; // Smallest first
                });
                break;
            default:
                // Default: date_desc
                sorted.sort((a, b) => {
                    const dateA = new Date(a.created_at || 0);
                    const dateB = new Date(b.created_at || 0);
                    return dateB - dateA;
                });
        }
        
        return sorted;
    }
    
    function sortAndDisplayImages(images) {
        const sorted = sortImages(images, currentSortBy);
        displayImages(sorted);
        // Update pagination if needed (for now, show all sorted images)
        $('#photoweb-gallery-pagination').hide();
    }
    
    // Refresh gallery
    $('#photoweb-refresh-gallery').on('click', function() {
        loadGallery(currentPage, currentFolderId);
    });
    
    // Pagination
    $(document).on('click', '#photoweb-gallery-pagination button', function() {
        currentPage = $(this).data('page');
        loadGallery(currentPage, currentFolderId);
    });
    
    // View image details
    $(document).on('click', '.view-image-details, .photoweb-gallery-image', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const imageId = $(this).data('image-id') || $(this).closest('.photoweb-gallery-item').data('image-id');
        if (imageId) {
            showImageDetails(imageId);
        }
    });
    
    // Copy URL
    $(document).on('click', '.copy-url', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const url = $(this).data('url');
        navigator.clipboard.writeText(url).then(function() {
            showToast('success', 'URL copied to clipboard!');
        }).catch(function() {
            showToast('error', 'Failed to copy URL. Please try again.');
        });
    });
    
    // Image Details Modal
    function showImageDetails(imageId) {
        const $modal = $('#photoweb-image-modal');
        const $modalBody = $('#photoweb-image-modal-body');
        
        $modal.show();
        $modalBody.html('<div class="photoweb-modal-loader"><span class="spinner is-active"></span><p>Loading image details...</p></div>');
        
        // Fetch image details from API
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_get_image_details',
                nonce: photowebData.nonce,
                image_id: imageId
            },
            success: function(response) {
                if (response.success && response.data) {
                    renderImageDetails(response.data);
                } else {
                    $modalBody.html('<div class="photoweb-modal-error"><p>Failed to load image details.</p></div>');
                }
            },
            error: function() {
                $modalBody.html('<div class="photoweb-modal-error"><p>An error occurred while loading image details.</p></div>');
            }
        });
    }
    
    function renderImageDetails(image) {
        const $modalBody = $('#photoweb-image-modal-body');
        
        // Format file size
        const fileSize = image.image_size_mb ? image.image_size_mb + ' MB' : (image.image_size ? formatFileSize(image.image_size) : 'N/A');
        
        // Format date
        const createdDate = image.created_at ? new Date(image.created_at).toLocaleString() : 'N/A';
        
        // Get URLs for different sizes
        const thumbUrl = image.image_thumb || '';
        const mediumUrl = image.image_medium || '';
        const fullUrl = image.image_full || image.image_medium || '';
        
        const html = '<div class="photoweb-image-details">' +
            '<div class="photoweb-image-details-preview">' +
            '<img src="' + mediumUrl + '" alt="' + escapeHtml(image.image_name) + '" id="photoweb-modal-image">' +
            '</div>' +
            '<div class="photoweb-image-details-info">' +
            '<div class="photoweb-detail-section">' +
            '<h3>Image Information</h3>' +
            '<table class="photoweb-details-table">' +
            '<tr><th>Name:</th><td>' + escapeHtml(image.image_name) + '</td></tr>' +
            '<tr><th>Code:</th><td><code>' + escapeHtml(image.image_code) + '</code></td></tr>' +
            '<tr><th>Size:</th><td>' + fileSize + '</td></tr>' +
            '<tr><th>Folder:</th><td>' + escapeHtml(image.folder_name || 'Default') + '</td></tr>' +
            '<tr><th>User ID:</th><td>' + (image.user_id || 'N/A') + '</td></tr>' +
            (image.account_id ? '<tr><th>Account ID:</th><td>' + escapeHtml(image.account_id) + '</td></tr>' : '') +
            '<tr><th>Created:</th><td>' + createdDate + '</td></tr>' +
            '</table>' +
            '</div>' +
            '<div class="photoweb-detail-section">' +
            '<h3>Image URLs</h3>' +
            '<div class="photoweb-url-group">' +
            '<label>Thumbnail:</label>' +
            '<div class="photoweb-url-input-group">' +
            '<input type="text" class="photoweb-url-input" value="' + escapeHtml(thumbUrl) + '" readonly>' +
            '<button type="button" class="button copy-url-btn" data-url="' + escapeHtml(thumbUrl) + '">📋 Copy</button>' +
            '</div>' +
            '</div>' +
            '<div class="photoweb-url-group">' +
            '<label>Medium:</label>' +
            '<div class="photoweb-url-input-group">' +
            '<input type="text" class="photoweb-url-input" value="' + escapeHtml(mediumUrl) + '" readonly>' +
            '<button type="button" class="button copy-url-btn" data-url="' + escapeHtml(mediumUrl) + '">📋 Copy</button>' +
            '</div>' +
            '</div>' +
            '<div class="photoweb-url-group">' +
            '<label>Full Size:</label>' +
            '<div class="photoweb-url-input-group">' +
            '<input type="text" class="photoweb-url-input" value="' + escapeHtml(fullUrl) + '" readonly>' +
            '<button type="button" class="button copy-url-btn" data-url="' + escapeHtml(fullUrl) + '">📋 Copy</button>' +
            '</div>' +
            '</div>' +
            '</div>' +
            '<div class="photoweb-detail-section">' +
            '<h3>Actions</h3>' +
            '<div class="photoweb-modal-actions">' +
            '<button type="button" class="button button-primary view-full-image" data-url="' + escapeHtml(fullUrl) + '">View Full Image</button>' +
            '<button type="button" class="button button-secondary photoweb-generate-shortcode" data-code="' + escapeHtml(image.image_code) + '" data-size="full">Generate Shortcode</button>' +
            '<button type="button" class="button button-secondary delete-image-modal" data-id="' + image.image_id + '">Delete Image</button>' +
            '</div>' +
            '</div>' +
            '<div class="photoweb-shortcode-generator" style="display: none; margin-top: 15px; padding: 15px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px;">' +
            '<h4>Shortcode Generator</h4>' +
            '<div class="photoweb-shortcode-options" style="margin-bottom: 10px;">' +
            '<label style="display: block; margin-bottom: 5px;">Size: <select id="shortcode-size" style="margin-left: 5px;"><option value="full">Full</option><option value="medium">Medium</option><option value="thumb">Thumbnail</option></select></label>' +
            '<label style="display: block; margin-bottom: 5px;"><input type="checkbox" id="shortcode-link"> Link to full image</label>' +
            '<label style="display: block; margin-bottom: 5px;">Width: <input type="text" id="shortcode-width" placeholder="e.g., 500px" style="margin-left: 5px; width: 150px;"></label>' +
            '<label style="display: block; margin-bottom: 5px;">Height: <input type="text" id="shortcode-height" placeholder="e.g., 300px" style="margin-left: 5px; width: 150px;"></label>' +
            '<label style="display: block; margin-bottom: 5px;">Alt Text: <input type="text" id="shortcode-alt" placeholder="Image description" style="margin-left: 5px; width: 200px;"></label>' +
            '</div>' +
            '<div class="photoweb-shortcode-output" style="margin-bottom: 10px;">' +
            '<label style="display: block; margin-bottom: 5px; font-weight: 600;">Generated Shortcode:</label>' +
            '<input type="text" id="shortcode-result" readonly style="width: 100%; padding: 8px; font-family: monospace; background: #fff; border: 1px solid #ccc;">' +
            '</div>' +
            '<button type="button" class="button button-primary photoweb-copy-shortcode">Copy Shortcode</button>' +
            '</div>' +
            '</div>' +
            '</div>';
        
        $modalBody.html(html);
    }
    
    // Close modal
    $(document).on('click', '.photoweb-modal-close, .photoweb-modal-overlay', function() {
        $('#photoweb-image-modal').hide();
    });
    
    // Copy URL from modal
    $(document).on('click', '.copy-url-btn', function(e) {
        e.preventDefault();
        const url = $(this).data('url');
        navigator.clipboard.writeText(url).then(function() {
            showToast('success', 'URL copied to clipboard!');
        }).catch(function() {
            showToast('error', 'Failed to copy URL. Please try again.');
        });
    });
    
    // View full image
    $(document).on('click', '.view-full-image', function() {
        const url = $(this).data('url');
        window.open(url, '_blank');
    });
    
    // Generate shortcode
    $(document).on('click', '.photoweb-generate-shortcode', function() {
        const $generator = $('.photoweb-shortcode-generator');
        $generator.slideToggle();
        updateShortcode();
    });
    
    function updateShortcode() {
        const code = $('.photoweb-generate-shortcode').data('code');
        const size = $('#shortcode-size').val();
        const link = $('#shortcode-link').is(':checked');
        const width = $('#shortcode-width').val();
        const height = $('#shortcode-height').val();
        const alt = $('#shortcode-alt').val();
        
        let shortcode = '[foldious_image code="' + escapeHtml(code) + '" size="' + size + '"';
        if (width) shortcode += ' width="' + escapeHtml(width) + '"';
        if (height) shortcode += ' height="' + escapeHtml(height) + '"';
        if (alt) shortcode += ' alt="' + escapeHtml(alt) + '"';
        if (link) shortcode += ' link="true"';
        shortcode += ']';
        
        $('#shortcode-result').val(shortcode);
    }
    
    $(document).on('change', '#shortcode-size, #shortcode-link, #shortcode-width, #shortcode-height, #shortcode-alt', updateShortcode);
    
    $(document).on('click', '.photoweb-copy-shortcode', function() {
        const shortcode = $('#shortcode-result').val();
        navigator.clipboard.writeText(shortcode).then(function() {
            showToast('success', 'Shortcode copied to clipboard!');
        }).catch(function() {
            // Fallback for older browsers
            $('#shortcode-result').select();
            document.execCommand('copy');
            showToast('success', 'Shortcode copied to clipboard!');
        });
    });
    
    // Delete image from modal
    $(document).on('click', '.delete-image-modal', function() {
        const imageId = $(this).data('id');
        const imageName = $('#photoweb-image-modal-body').find('table tr:first-child td').text() || 'this image';
        
        if (!confirm('Are you sure you want to delete "' + imageName + '"?')) {
            return;
        }
        
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_delete_image',
                nonce: photowebData.nonce,
                image_id: imageId
            },
            success: function(response) {
                if (response.success) {
                    showToast('success', 'Image deleted successfully');
                    $('#photoweb-image-modal').hide();
                    // Remove from selected if it was selected
                    selectedImageIds = selectedImageIds.filter(id => id !== imageId);
                    updateBulkActionsUI();
                    loadGallery(currentPage, currentFolderId);
                } else {
                    showToast('error', 'Failed to delete image: ' + (response.data.message || 'Unknown error'));
                }
            },
            error: function() {
                showToast('error', 'An error occurred while deleting the image');
            }
        });
    });
    
    // Bulk selection
    $(document).on('change', '.photoweb-image-checkbox', function() {
        const imageId = parseInt($(this).data('image-id'));
        const isChecked = $(this).is(':checked');
        
        if (isChecked) {
            if (!selectedImageIds.includes(imageId)) {
                selectedImageIds.push(imageId);
            }
        } else {
            selectedImageIds = selectedImageIds.filter(id => id !== imageId);
        }
        
        updateBulkActionsUI();
    });
    
    // Select all images
    $('#photoweb-select-all-btn').on('click', function() {
        const allVisibleIds = [];
        $('.photoweb-gallery-item').each(function() {
            const imageId = parseInt($(this).data('image-id'));
            allVisibleIds.push(imageId);
            if (!selectedImageIds.includes(imageId)) {
                selectedImageIds.push(imageId);
            }
            $(this).find('.photoweb-image-checkbox').prop('checked', true);
            $(this).addClass('selected');
        });
        updateBulkActionsUI();
    });
    
    // Deselect all
    $('#photoweb-bulk-deselect').on('click', function() {
        selectedImageIds = [];
        $('.photoweb-image-checkbox').prop('checked', false);
        $('.photoweb-gallery-item').removeClass('selected');
        updateBulkActionsUI();
    });
    
    // Bulk delete
    $('#photoweb-bulk-delete').on('click', function() {
        if (selectedImageIds.length === 0) {
            showToast('warning', 'Please select at least one image to delete');
            return;
        }
        
        if (!confirm(`Are you sure you want to delete ${selectedImageIds.length} image(s)?`)) {
            return;
        }
        
        const $btn = $(this);
        $btn.prop('disabled', true).html('<span class="spinner is-active"></span> Deleting...');
        
        // Delete images one by one (or we can create a bulk delete API)
        let deletedCount = 0;
        let failedCount = 0;
        const total = selectedImageIds.length;
        
        const deletePromises = selectedImageIds.map(function(imageId) {
            return $.ajax({
                url: photowebData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'photoweb_delete_image',
                    nonce: photowebData.nonce,
                    image_id: imageId
                }
            }).then(function(response) {
                if (response.success) {
                    deletedCount++;
                    return { success: true, imageId: imageId };
                } else {
                    failedCount++;
                    return { success: false, imageId: imageId };
                }
            }).catch(function() {
                failedCount++;
                return { success: false, imageId: imageId };
            });
        });
        
        Promise.all(deletePromises).then(function(results) {
            // Remove successfully deleted images from DOM
            results.forEach(function(result) {
                if (result.success) {
                    $('.photoweb-gallery-item[data-image-id="' + result.imageId + '"]').fadeOut(function() {
                        $(this).remove();
                    });
                }
            });
            
            // Update selected array
            selectedImageIds = results.filter(r => !r.success).map(r => r.imageId);
            
            if (deletedCount > 0) {
                showToast('success', `${deletedCount} image(s) deleted successfully${failedCount > 0 ? ', ' + failedCount + ' failed' : ''}`);
            } else {
                showToast('error', 'Failed to delete images. Please try again.');
            }
            
            // Reload gallery if all visible images are deleted
            if ($('#photoweb-gallery-grid').children().length === 0) {
                loadGallery(currentPage, currentFolderId);
            } else {
                updateBulkActionsUI();
            }
            
            $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Delete Selected');
        });
    });
    
    function updateBulkActionsUI() {
        const count = selectedImageIds.length;
        const $bulkActions = $('#photoweb-bulk-actions');
        const $selectAllBtn = $('#photoweb-select-all-btn');
        
        if (count > 0) {
            $('#photoweb-selected-count').text(count);
            $bulkActions.show();
            $selectAllBtn.show();
        } else {
            $bulkActions.hide();
            $selectAllBtn.hide();
        }
        
        // Update item selection state
        $('.photoweb-gallery-item').each(function() {
            const imageId = parseInt($(this).data('image-id'));
            if (selectedImageIds.includes(imageId)) {
                $(this).addClass('selected');
            } else {
                $(this).removeClass('selected');
            }
        });
    }
    
    // Click on image item to toggle selection (when in selection mode)
    $(document).on('click', '.photoweb-gallery-item', function(e) {
        // Don't trigger if clicking on buttons or checkbox
        if ($(e.target).closest('.photoweb-gallery-item-actions, .photoweb-image-checkbox, .photoweb-gallery-item-checkbox').length > 0) {
            return;
        }
        
        const $checkbox = $(this).find('.photoweb-image-checkbox');
        $checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
    });
    
    // Delete image
    $(document).on('click', '.delete-image', function() {
        const imageId = $(this).data('id');
        const $item = $(this).closest('.photoweb-gallery-item');
        const imageName = $item.find('h4').text() || 'this image';
        
        if (!confirm('Are you sure you want to delete "' + imageName + '"?')) {
            return;
        }
        
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_delete_image',
                nonce: photowebData.nonce,
                image_id: imageId
            },
            success: function(response) {
                if (response.success) {
                    showToast('success', 'Image deleted successfully');
                    // Remove from selected if it was selected
                    selectedImageIds = selectedImageIds.filter(id => id !== imageId);
                    updateBulkActionsUI();
                    $item.fadeOut(function() {
                        $(this).remove();
                        if ($('#photoweb-gallery-grid').children().length === 0) {
                            loadGallery(currentPage, currentFolderId);
                        }
                    });
                } else {
                    showToast('error', 'Failed to delete image: ' + (response.data.message || 'Unknown error'));
                }
            },
            error: function() {
                showToast('error', 'An error occurred while deleting the image');
            }
        });
    });
    
    // Keyboard Shortcuts
    let keyboardShortcutsEnabled = true;
    
    $(document).on('keydown', function(e) {
        // Don't trigger shortcuts when typing in inputs, textareas, or when modals are open
        if ($(e.target).is('input, textarea, select') || $('.photoweb-modal').is(':visible')) {
            return;
        }
        
        // Check for modifier keys
        const isCtrl = e.ctrlKey || e.metaKey;
        const isShift = e.shiftKey;
        const isAlt = e.altKey;
        
        // Ctrl/Cmd + U: Go to Upload page
        if (isCtrl && e.key === 'u' && !isShift && !isAlt) {
            e.preventDefault();
            window.location.href = '?page=photoweb-upload';
            return;
        }
        
        // Ctrl/Cmd + G: Go to Gallery page
        if (isCtrl && e.key === 'g' && !isShift && !isAlt) {
            e.preventDefault();
            window.location.href = '?page=photoweb-gallery';
            return;
        }
        
        // Ctrl/Cmd + D: Go to Dashboard page
        if (isCtrl && e.key === 'd' && !isShift && !isAlt) {
            e.preventDefault();
            window.location.href = '?page=photoweb-dashboard';
            return;
        }
        
        // Gallery-specific shortcuts
        if ($('#photoweb-gallery-grid').length > 0) {
            // Ctrl/Cmd + A: Select all visible images
            if (isCtrl && e.key === 'a' && !isShift && !isAlt) {
                e.preventDefault();
                if ($('#photoweb-select-all-btn').is(':visible')) {
                    $('#photoweb-select-all-btn').click();
                }
                return;
            }
            
            // Delete/Backspace: Delete selected images
            if ((e.key === 'Delete' || e.key === 'Backspace') && !isCtrl && !isShift && !isAlt) {
                if (selectedImageIds.length > 0) {
                    e.preventDefault();
                    $('#photoweb-bulk-delete').click();
                }
                return;
            }
            
            // Escape: Deselect all
            if (e.key === 'Escape' && !isCtrl && !isShift && !isAlt) {
                if (selectedImageIds.length > 0) {
                    e.preventDefault();
                    $('#photoweb-bulk-deselect').click();
                }
                return;
            }
            
            // Ctrl/Cmd + F: Focus search input
            if (isCtrl && e.key === 'f' && !isShift && !isAlt) {
                e.preventDefault();
                $('#gallery-search').focus();
                return;
            }
            
            // Arrow keys: Navigate between images (if no input is focused)
            if (!$(e.target).is('input, textarea, select')) {
                if (e.key === 'ArrowRight' || e.key === 'ArrowLeft') {
                    e.preventDefault();
                    navigateGallery(e.key === 'ArrowRight' ? 1 : -1);
                    return;
                }
            }
        }
        
        // Upload page shortcuts
        if ($('#photoweb-upload-form').length > 0) {
            // Ctrl/Cmd + Enter: Submit upload form
            if (isCtrl && e.key === 'Enter' && !isShift && !isAlt) {
                e.preventDefault();
                if (selectedFiles.length > 0 && !$('#photoweb-upload-btn').prop('disabled')) {
                    $('#photoweb-upload-form').submit();
                }
                return;
            }
        }
    });
    
    // Gallery navigation with arrow keys
    function navigateGallery(direction) {
        const $items = $('.photoweb-gallery-item');
        if ($items.length === 0) return;
        
        // Find currently focused item or first item
        let currentIndex = -1;
        $items.each(function(index) {
            if ($(this).hasClass('keyboard-focused')) {
                currentIndex = index;
                return false;
            }
        });
        
        if (currentIndex === -1) {
            currentIndex = 0;
        } else {
            currentIndex += direction;
            if (currentIndex < 0) currentIndex = $items.length - 1;
            if (currentIndex >= $items.length) currentIndex = 0;
        }
        
        // Remove previous focus
        $items.removeClass('keyboard-focused');
        
        // Add focus to new item
        const $newItem = $items.eq(currentIndex);
        $newItem.addClass('keyboard-focused');
        
        // Scroll into view
        const offset = $newItem.offset().top - $(window).scrollTop();
        if (offset < 100 || offset > $(window).height() - 100) {
            $('html, body').animate({
                scrollTop: $newItem.offset().top - 200
            }, 300);
        }
        
        // Enter key on focused item: Open details modal
        $(document).off('keydown.navigateGallery').on('keydown.navigateGallery', function(e) {
            if (e.key === 'Enter' && !isCtrl && !isShift && !isAlt) {
                const $focused = $('.photoweb-gallery-item.keyboard-focused');
                if ($focused.length > 0) {
                    e.preventDefault();
                    const imageId = $focused.data('image-id');
                    if (imageId) {
                        showImageDetails(imageId);
                    }
                }
            }
        });
    }
    
    // Show keyboard shortcuts help
    function showKeyboardShortcutsHelp() {
        const shortcuts = [
            { key: 'Ctrl/Cmd + U', action: 'Go to Upload page' },
            { key: 'Ctrl/Cmd + G', action: 'Go to Gallery page' },
            { key: 'Ctrl/Cmd + D', action: 'Go to Dashboard page' },
            { key: 'Ctrl/Cmd + A', action: 'Select all images (Gallery)' },
            { key: 'Delete/Backspace', action: 'Delete selected images (Gallery)' },
            { key: 'Escape', action: 'Deselect all images (Gallery)' },
            { key: 'Ctrl/Cmd + F', action: 'Focus search (Gallery)' },
            { key: 'Arrow Keys', action: 'Navigate between images (Gallery)' },
            { key: 'Enter', action: 'Open image details (Gallery, when focused)' },
            { key: 'Ctrl/Cmd + Enter', action: 'Submit upload form (Upload page)' }
        ];
        
        let html = '<div class="photoweb-shortcuts-help">';
        html += '<h3>Keyboard Shortcuts</h3>';
        html += '<table class="photoweb-shortcuts-table">';
        shortcuts.forEach(function(shortcut) {
            html += '<tr><td><kbd>' + escapeHtml(shortcut.key) + '</kbd></td><td>' + escapeHtml(shortcut.action) + '</td></tr>';
        });
        html += '</table>';
        html += '<button type="button" class="button button-primary photoweb-close-shortcuts">Close</button>';
        html += '</div>';
        
        const $modal = $('<div class="photoweb-modal"><div class="photoweb-modal-overlay"></div><div class="photoweb-modal-content photoweb-shortcuts-modal">' + html + '</div></div>');
        $('body').append($modal);
        $modal.show();
        
        $modal.find('.photoweb-modal-overlay, .photoweb-close-shortcuts').on('click', function() {
            $modal.fadeOut(function() {
                $(this).remove();
            });
        });
    }
    
    // Add keyboard shortcuts help button to dashboard
    if ($('#photoweb-dashboard').length > 0) {
        const $helpBtn = $('<button type="button" class="button" id="photoweb-shortcuts-help" style="margin-left: 10px;"><span class="dashicons dashicons-editor-code"></span> Keyboard Shortcuts</button>');
        $('.wrap h1').after($helpBtn);
        $('#photoweb-shortcuts-help').on('click', showKeyboardShortcutsHelp);
    }
    
    // Disconnect account
    $('#photoweb-disconnect-btn').on('click', function() {
        if (!confirm('Are you sure you want to disconnect your Foldious account?')) {
            return;
        }
        
        // Clear options
        $.ajax({
            url: photowebData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'photoweb_disconnect',
                nonce: photowebData.nonce
            },
            success: function() {
                window.location.reload();
            }
        });
    });
    
    // Helper functions
    function showMessage($element, message, type, allowHtml = false) {
        $element.removeClass('success error')
                .addClass(type);
        
        if (allowHtml) {
            $element.html(message);
        } else {
            $element.text(message);
        }
        
        $element.show();
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Toast Notification System
    function showToast(type, message, duration = 4000) {
        // Create toast container if it doesn't exist
        let $toastContainer = $('#photoweb-toast-container');
        if ($toastContainer.length === 0) {
            $toastContainer = $('<div id="photoweb-toast-container"></div>');
            $('body').append($toastContainer);
        }
        
        // Create toast element
        const $toast = $('<div class="photoweb-toast photoweb-toast-' + type + '">' +
            '<div class="photoweb-toast-icon"></div>' +
            '<div class="photoweb-toast-message">' + escapeHtml(message) + '</div>' +
            '<button class="photoweb-toast-close" aria-label="Close">×</button>' +
            '</div>');
        
        // Add icon based on type
        const icons = {
            'success': '✓',
            'error': '✕',
            'warning': '⚠',
            'info': 'ℹ'
        };
        $toast.find('.photoweb-toast-icon').text(icons[type] || icons.info);
        
        // Append to container
        $toastContainer.append($toast);
        
        // Trigger animation
        setTimeout(function() {
            $toast.addClass('show');
        }, 10);
        
        // Auto remove
        const autoRemove = setTimeout(function() {
            removeToast($toast);
        }, duration);
        
        // Manual close
        $toast.find('.photoweb-toast-close').on('click', function() {
            clearTimeout(autoRemove);
            removeToast($toast);
        });
        
        // Click to dismiss
        $toast.on('click', function(e) {
            if (!$(e.target).hasClass('photoweb-toast-close')) {
                clearTimeout(autoRemove);
                removeToast($toast);
            }
        });
    }
    
    function removeToast($toast) {
        $toast.removeClass('show');
        setTimeout(function() {
            $toast.remove();
        }, 300);
    }
    
    // Get actionable error messages
    function getActionableErrorMessage(errorData) {
        if (!errorData) {
            return 'Upload failed. Please try again.';
        }
        
        const message = errorData.message || 'Upload failed';
        let actionableMessage = message;
        
        // Add actionable steps based on error type
        if (message.includes('User not authenticated') || message.includes('user_id')) {
            actionableMessage += '<br><strong>Action:</strong> Please reconnect your Foldious account in Settings.';
        } else if (message.includes('size') || message.includes('too large')) {
            actionableMessage += '<br><strong>Action:</strong> Please select smaller images (max 10MB per image).';
        } else if (message.includes('type') || message.includes('format')) {
            actionableMessage += '<br><strong>Action:</strong> Please select only JPG, PNG, GIF, or WebP images.';
        } else if (message.includes('network') || message.includes('connection')) {
            actionableMessage += '<br><strong>Action:</strong> Check your internet connection and try again.';
        } else if (message.includes('API') || message.includes('server')) {
            actionableMessage += '<br><strong>Action:</strong> The server may be temporarily unavailable. Please try again in a few moments.';
        } else {
            actionableMessage += '<br><strong>Tip:</strong> You can retry failed uploads using the "Retry Failed Uploads" button.';
        }
        
        return actionableMessage;
    }
    
})(jQuery);

